function [varargout] = InitPar(X,F,Options)
% Find initial estimates for a rank F PARAFAC model of X
% 
% SYNTAX
% [A,B,C] = InitPar (X,F,Options);
% 
% INPUT 
% X      : 3-way array
% F      : model's rank
% Options: see ParOptions 
%          The 'initialisation' field is a structure with the following fields
%              'method': six type of initialisation are available
%                        'orth'  : ten starts with orthogonal random matrices. The best is picked after 'iter' iterations 
%                        'random': ten starts with random matrices. The best is picked after 'iter' iterations 
%                        'svd'   : the matrices are obtained via SVD  
%                        'swatld': the SWATLD algorithm is used 
%                        'dltd'  : the DTLD algorithm is used 
%                        'best'  : tries all the previous initalisations and picks the best one 
%               'iter' : max number of iterations for the iterative methods
%               'tol'  : tolerance (i.e. relative fit decrease) for convergence in the iterative methods
% 
% OUTPUT
% A,B,C: initial parameter estimates
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Denmark 
% 
% Last modified: 10-Jan-2005 15:23
% 
% Contact: Giorgio Tomasi, gt@kvl.dk, Rasmus Bro, rb@kvl.dk 
%

% Some initialisations
Fit0 = inf;
dimX = size(X);

% Check inputs
if ndims(X) ~= 3
    error('This implementation cannot handle arrays of order higher than 3')
end
if strcmpi(Options.initialisation.method,'dtld') & ~exist('dtld.m','file')
    warning('DTLD.m is not on the path. Using SVD as initialisation method')
    Options.initialisation.method = 'svd';
end
if any(strcmpi({'random','orth'},Options.initialisation.method)) & ~exist('PARAFAC.m','file')
    warning('PARAFAC.m is not on the path. Using SVD as initialisation method')
    Options.initialisation.method = 'svd';
end
% Turn iteration and diagnostic display off
Options = ParOptions(Options,'display','none','diagnostics','off');

switch Options.initialisation.method % Initialise according to the chosen method
    case {'random','orth'}
        
        for i=1:10
            
            B = rand(dimX(2),F);
            C = rand(dimX(3),F);
            if strcmpi(Options.initialisation.method,'random')
                A = reshape(X,dimX(1),dimX(2) * dimX(3)) * kr(C,B) * pinv((B'*B) .* (C'*C));
            else
                B = orth(B);
                C = orth(C);
                A = reshape(X,dimX(1),dimX(2) * dimX(3)) * kr(C,B);
            end
            %run Options.initialisation.maxiter iterations with ALS
            OptionsALS    = [Options.initialisation.tol,0,0,0,NaN,Options.initialisation.maxiter];
            [Fac,nil,fit] = parafac(X,F,OptionsALS,[],{A,B,C}); 
            if fit < Fit0
                [varargout{1:ndims(X)}] = deal(A,B,C);
                Fit0                    = fit;
            end
            
        end
        
    case 'svd'
        % Uses eig and projection (since it is only for initialisation extreme accuracy is not necessary)
        [dimX,Order]  = sort(dimX);
        X             = permute(X,Order);
        Order(Order)  = 1:length(dimX);
        X             = reshape(X,dimX(1),prod(dimX(2:end)));
        for n = 1:length(dimX)

            if size(X,2) > size(X,1)
                [U,S] = eig(X * X');
                U     = U(:,end - F + 1:end);
            else
                [U,S] = eig(X' * X);
                U     = U(:,end - F + 1:end);
                U     = X * (U * (S(end - F + 1:end,end - F + 1:end)^-0.5));
            end
            if size(U,2) < F
                U(:,end + 1:F) = orth(rand(size(U,1),F - size(U,2)));
            end
            if n == length(dimX)
                varargout{n} = U * S(end - F + 1:end,end - F + 1:end)^0.5;
            else
                varargout{n} = U;
                dimX(1)      = F;
                dimX         = dimX([2:end,1]);
                X            = reshape(X' * varargout{n},dimX(1),prod(dimX(2:end)));
            end
            
        end
        varargout = varargout(Order);
        
    case 'dtld'
        [varargout{1:3}] = dtld(X,F,0);
        
    case 'swatld'
        Opt         = ParOptions(Options,'cc_maxiter',Options.initialisation.maxiter,'cc_relfit',Options.initialisation.tol);
        [varargout] = swatld(X,F,Options,rand(dimX(1),F),rand(dimX(2),F),rand(dimX(3),F));
        
    case 'best'
        % Try the different initialisations
        Options                       = ParOptions(Options,'init_method','orth');
        [Factors{1,1:ndims(X)}]       = InitPar(X,F,Options);
        Fit(1)                        = tss(X - nmodel(Factors(1,1:ndims(X))),false);
        Options                       = ParOptions(Options,'init_method','random');
        [Factors{2,1:ndims(X)}]       = InitPar(X,F,Options);
        Fit(2)                        = tss(X - nmodel(Factors(2,1:ndims(X))),false);
        Options                       = ParOptions(Options,'init_method','svd');
        [Factors{3,1:ndims(X)}]       = InitPar(X,F,Options);
        Fit(3)                        = tss(X - nmodel(Factors(3,1:ndims(X))),false);
        Options                       = ParOptions(Options,'init_method','swatld');
        [Factors{4,1:ndims(X)}]       = InitPar(X,F,Options);
        Fit(4)                        = tss(X - nmodel(Factors(4,1:ndims(X))),false);
        Options                       = ParOptions(Options,'init_method','dtld');
        [Factors{5,1:ndims(X)}]       = InitPar(X,F,Options);
        Fit(5)                        = tss(X - nmodel(Factors(5,1:ndims(X))),false);
        % Pick the initialisation with the lowest loss function value
        [Nil,Best] = min(Fit);
        varargout  = Factors(Best,:);
        
    otherwise 
        error('There is not such an initialisation')
        
end